#!/bin/bash

###############################################################################
SYSCTL="/sbin/sysctl -w"
SYSCTLFILE="/etc/sysctl.conf"
# To echo the value directly to the /proc file instead
# SYSCTL=""
# Don't make the changes permanently
# SYSCTLFILE=""

###############################################################################
# Get time stamp for a backup
day_time=$(date "+%Y%m%d_%H%M%S")

if [ "$SYSCTLFILE" != "" ]; then
  cp $SYSCTLFILE $SYSCTLFILE.bck-$day_time
  if [ "$?" != "0" ]; then
    echo "ERROR: cannot make a backup. Exiting ..."
    exit 1
  fi
fi

###############################################################################
# predefined function

make_change(){
  # This option logs packets from impossible addresses.
  if [ "$SYSCTL" == "" ]; then
      echo "$SysValue" > $ProcFile
  else
      $SYSCTL $SysEntry="$SysValue"
  fi

  if [ "$SYSCTLFILE" != "" ]; then
      grep "$SysEntry \?=" $SYSCTLFILE
      if [ "$?" == "0" ]; then
          sed -i.tmp "s/$SysEntry \?=.*/$SysEntry = $SysValue/" $SYSCTLFILE
      else
          echo "$SysEntry = $SysValue" >>$SYSCTLFILE
      fi
  fi
}
###############################################################################

#
# Kernel Parameter Configuration
#
# See http://ipsysctl-tutorial.frozentux.net/chunkyhtml/index.html
# for a detailed tutorial on sysctl and the various settings
# available.

# Required to enable IPv4 forwarding.
# Redhat users can try setting FORWARD_IPV4 in /etc/sysconfig/network to true
# Alternatively, it can be set in /etc/sysctl.conf
ProcFile="/proc/sys/net/ipv4/ip_forward"
SysEntry="net.ipv4.ip_forward"
SysValue="1"
make_change

# This enables dynamic address hacking.
# This may help if you have a dynamic IP address \(e.g. slip, ppp, dhcp\).
ProcFile="/proc/sys/net/ipv4/ip_dynaddr"
SysEntry="net.ipv4.ip_dynaddr"
SysValue="1"
#make_change


# This enables SYN flood protection.
# The SYN cookies activation allows your system to accept an unlimited
# number of TCP connections while still trying to give reasonable
# service during a denial of service attack.
ProcFile="/proc/sys/net/ipv4/tcp_syncookies"
SysEntry="net.ipv4.tcp_syncookies"
SysValue="1"
make_change

# This enables source validation by reversed path according to RFC1812.
# In other words, did the response packet originate from the same interface
# through which the source packet was sent?  It's recommended for single-homed
# systems and routers on stub networks.  Since those are the configurations
# this firewall is designed to support, I turn it on by default.
# Turn it off if you use multiple NICs connected to the same network.
ProcFile="/proc/sys/net/ipv4/conf/all/rp_filter"
SysEntry="net.ipv4.conf.all.rp_filter"
SysValue="1"
make_change

# This option allows a subnet to be firewalled with a single IP address.
# It's used to build a DMZ.  Since that's not a focus of this firewall
# script, it's not enabled by default, but is included for reference.
# See: http://www.sjdjweis.com/linux/proxyarp/ 
ProcFile="/proc/sys/net/ipv4/conf/all/proxy_arp"
SysEntry="net.ipv4.conf.all.proxy_arp"
SysValue="1"
#make_change

# The following kernel settings were suggested by Alex Weeks. Thanks!

# This kernel parameter instructs the kernel to ignore all ICMP
# echo requests sent to the broadcast address.  This prevents
# a number of smurfs and similar DoS nasty attacks.
ProcFile="/proc/sys/net/ipv4/icmp_echo_ignore_broadcasts"
SysEntry="net.ipv4.icmp_echo_ignore_broadcasts"
SysValue="1"
make_change

# This option can be used to accept or refuse source routed
# packets.  It is usually on by default, but is generally
# considered a security risk.  This option turns it off.
ProcFile="/proc/sys/net/ipv4/conf/all/accept_source_route"
SysEntry="net.ipv4.conf.all.accept_source_route"
SysValue="0"
make_change

# This option can disable ICMP redirects.  ICMP redirects
# are generally considered a security risk and shouldn't be
# needed by most systems using this generator.
ProcFile="/proc/sys/net/ipv4/conf/all/accept_redirects"
SysEntry="net.ipv4.conf.all.accept_redirects"
SysValue="0"
#make_change

# However, we'll ensure the secure_redirects option is on instead.
# This option accepts only from gateways in the default gateways list.
ProcFile="/proc/sys/net/ipv4/conf/all/secure_redirects"
SysEntry="net.ipv4.conf.all.secure_redirects"
SysValue="1"
make_change

# This option logs packets from impossible addresses.
ProcFile="/proc/sys/net/ipv4/conf/all/log_martians"
SysEntry="net.ipv4.conf.all.log_martians"
SysValue="1"
make_change
