# Copyright 1999-2024 Gentoo Authors
# Distributed under the terms of the GNU General Public License v2

EAPI=8

inherit flag-o-matic xdg-utils

DESCRIPTION="A portable, bytecode-compiled implementation of Common Lisp"
HOMEPAGE="https://clisp.sourceforge.io/"
SRC_URI="mirror://gentoo/${P}.tar.bz2"

LICENSE="GPL-2+"
SLOT="2/8"
KEYWORDS="~alpha amd64 ~mips ppc ppc64 ~riscv sparc x86"
IUSE="hyperspec X berkdb dbus fastcgi gdbm gtk +pcre postgres +readline svm threads +unicode +zlib"
# "jit" disabled ATM

RDEPEND=">=dev-lisp/asdf-2.33-r3
	virtual/libcrypt:=
	virtual/libiconv
	>=dev-libs/libsigsegv-2.10
	>=dev-libs/ffcall-1.10
	dbus? ( sys-apps/dbus )
	fastcgi? ( dev-libs/fcgi )
	gdbm? ( sys-libs/gdbm:0= )
	gtk? ( >=x11-libs/gtk+-2.10:2 >=gnome-base/libglade-2.6 )
	postgres? ( >=dev-db/postgresql-8.0:* )
	readline? ( >=sys-libs/readline-7.0:0= )
	pcre? ( dev-libs/libpcre:3 )
	svm? ( sci-libs/libsvm )
	zlib? ( sys-libs/zlib )
	X? ( x11-libs/libXpm )
	hyperspec? ( dev-lisp/hyperspec )
	berkdb? ( sys-libs/db:4.8 )"

DEPEND="${RDEPEND}
	X? ( x11-base/xorg-proto x11-misc/imake )"

enable_modules() {
	[[ $# = 0 ]] && die "${FUNCNAME[0]} must receive at least one argument"
	for m in "$@" ; do
		einfo "enabling module $m"
		myconf+=" --with-module=${m}"
	done
}

BUILDDIR="builddir"

# modules not enabled:
#  * berkdb: must figure out a way to make the configure script pick up the
#            currect version of the library and headers
#  * dirkey: fails to compile, requiring windows.h, possibly wrong #ifdefs
#  * matlab, netica: not in portage
#  * oracle: can't install oracle-instantclient

src_prepare() {
	# More than -O1 breaks alpha
	if use alpha; then
		sed -i -e 's/-O2//g' src/makemake.in || die
	fi
	eapply "${FILESDIR}"/"${P}"-after_glibc_cfree_bdb.patch
	eapply_user

	xdg_environment_reset
}

src_configure() {
	# -Werror=lto-type-mismatch
	# https://bugs.gentoo.org/856103
	# https://gitlab.com/gnu-clisp/clisp/-/issues/49
	filter-lto

	# We need this to build on alpha
	if use alpha; then
		replace-flags -O? -O1
	fi

	if use x86; then
		append-flags -falign-functions=4
	fi

	# built-in features
	local myconf="--with-ffcall --without-dynamic-modules"
#    There's a problem with jit_allocai function
#    if use jit; then
#        myconf+=" --with-jitc=lightning"
#    fi
	if use threads; then
		myconf+=" --with-threads=POSIX_THREADS"
	fi

	# default modules
	enable_modules rawsock
	# optional modules
	use elibc_glibc && enable_modules bindings/glibc
	use X && enable_modules clx/new-clx
	if use postgres; then
		enable_modules postgresql
		append-cppflags -I$(pg_config --includedir)
	fi
	if use berkdb; then
		enable_modules berkeley-db
		append-cppflags -I"${EPREFIX}"/usr/include/db4.8
	fi
	use dbus && enable_modules dbus
	use fastcgi && enable_modules fastcgi
	use gdbm && enable_modules gdbm
	use gtk && enable_modules gtk2
	use pcre && enable_modules pcre
	use svm && enable_modules libsvm
	use zlib && enable_modules zlib

	if use hyperspec; then
		CLHSROOT="file:///${EPREFIX}/usr/share/doc/hyperspec/HyperSpec/"
	else
		CLHSROOT="http://www.lispworks.com/reference/HyperSpec/"
	fi

	# configure chokes on --sysconfdir option
	local configure="./configure --prefix=${EPREFIX}/usr --enable-portability \
		  --libdir=${EPREFIX}/usr/$(get_libdir) $(use_with readline) $(use_with unicode) \
		  ${myconf} --hyperspec=${CLHSROOT} ${BUILDDIR}"
	einfo "${configure}"
	${configure} || die "./configure failed"

	IMPNOTES="file://${EPREFIX}/usr/share/doc/${PN}-${PVR}/html/impnotes.html"
	sed -i "s,http://clisp.cons.org/impnotes/,${IMPNOTES},g" \
		"${BUILDDIR}"/config.lisp || die "Cannot fix link to implementation notes"
}

src_compile() {
	export VARTEXFONTS="${T}"/fonts
	cd "${BUILDDIR}" || die
	# parallel build fails
	emake -j1
}

src_install() {
	pushd "${BUILDDIR}" || die
	make DESTDIR="${D}" prefix="${EPREFIX}"/usr install-bin || die "Installation failed"
	doman clisp.1
	dodoc ../SUMMARY README* ../src/NEWS ../unix/MAGIC.add ../ANNOUNCE
	popd || die
	dodoc doc/{CLOS-guide,LISP-tutorial}.txt
	docinto html
	dodoc doc/impnotes.{css,html} doc/regexp.html doc/clisp.png
}
